import asyncio
import importlib
import time
from datetime import datetime
from py_pli.pylib import EndPointType
from py_pli.pylib import URPCFunctions
from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal


# From the terminal use the names defined in this function to access the endpoints directly.
def __getattr__(name):
    if name in nodes:
        return get_node_endpoint(name)
    if name in movers:
        return get_mover_endpoint(name)
    if name == 'corexy':
        return get_corexy_endpoint()
    if name == 'measurement' or name == 'meas':
        return get_measurement_endpoint()
    if name in serial_endpoints:
        return get_serial_endpoint(name)
    if name in 'sys':
        return get_system_control_endpoint()
    if name in 'fan':
        return get_fan_control_endpoint()
    if name in temperature_controls:
        return get_temperature_control_endpoint(name)
    raise AttributeError(f"module '{__name__}' has no attribute '{name}'")


# Helper function to create endpoints that are not supported by the PyRunner yet.
def create_endpoint(can_id, endpoint_class_name):
    endpoint_creator = URPCFunctions.instance.endPointCreator

    endpoint_module = importlib.import_module('urpc.' + endpoint_class_name.lower())
    endpoint_class = getattr(endpoint_module, endpoint_class_name)

    endpoint = endpoint_class(can_id, endpoint_creator.transmitterQueueParent, endpoint_creator.receptionQueueChild, endpoint_creator.iso15765xSend, endpoint_creator.event_loop)

    endpoint_creator.creationEvent.clear()
    endpoint_creator.createEndpointQueueParent.send(can_id)
    endpoint_creator.creationEvent.wait()
    endpoint_creator.creationEvent.clear()

    endpoint_creator.urpcFunctions.endPointsDic[can_id] = endpoint

    return endpoint


# Float to Q31 format. Value must be [-1.0, +1.0).
def float_to_q31(value):
    q31 = round(value * 2**31)
    if q31 > 2**31 - 1:
        q31 = 2**31 - 1
    if q31 < -2**31:
        q31 = -2**31
    return int(q31)


# Q31 format to Float.
def q31_to_float(q31):
    value = q31 / 2**31
    return value


# Node Endpoint ########################################################################################################

nodes = {
    'eef'   : {'id':0x0008, 'delay':5}, 
#    'eef2'  : {'id':0x00E0, 'delay':5},
    'mc6'   : {'id':0x0010, 'delay':1},
    'fmb'   : {'id':0x0018, 'delay':1},
#    'pmc'   : {'id':0x00F0, 'delay':1},
#    'pmc1'  : {'id':0x00F1, 'delay':1},
#    'pmc2'  : {'id':0x00F2, 'delay':1},
#    'pmc3'  : {'id':0x00F3, 'delay':1},
}

def get_node_endpoint(node_name):
    can_id = nodes[node_name]['id']
    node = URPCFunctions.instance.endPointsDic.get(can_id)
    if node is None:
        node = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.NODE)
    return node

async def start_firmware(node_name):
    node = get_node_endpoint(node_name)
    if (await node.GetFirmwareInfo(timeout=1))[0] != 0:
        await node.StartApplication(timeout=1)
        await asyncio.sleep(nodes[node_name]['delay'])
        if (await node.GetFirmwareInfo(timeout=1))[0] != 0:
            raise Exception(f"Failed to start the firmware.")


async def get_din(node_name):
    start_firmware(node_name)
    node = get_node_endpoint(node_name)
    din = (await node.GetDigitalInput(1))[0]
    await asyncio.sleep(0.5)
    print(f"      21098765432109876543210987654321")
    print(f"din = {din:032b}")

async def get_ain(node_name, input_number):
    node = get_node_endpoint(node_name)
    q31 = (await node.GetAnalogInput(input_number, timeout=1))[0]
    return q31_to_float(q31)

# scale analog value
def analogscale(aval):
    return aval / 4096
#    return aval * 524288


# Mover Endpoint ########################################################################################################

movers = {
    'as1'       : {'id':0x010A},  # Aperture Slider 1         (266)
    'as2'       : {'id':0x010B},  # Aperture Slider 2         (267)
    'fms'       : {'id':0x0110},  # Filder Module Slider      (272)
    'usfm'      : {'id':0x0111},  # US Lum Focus Mover        (273)
    'fm'        : {'id':0x0112},  # Focus Mover               (274)
    'bld'       : {'id':0x0113},  # Bottom Light Director     (275)
    'pd'        : {'id':0x0114},  # Plate Door                (276)
    'els'       : {'id':0x0115},  # Excitation Light Selector (277)
#    'zdrive'    : {'id':0x01F0},  # Z-Drive of PMC            (496)
#    'pipettor'  : {'id':0x02F0},  # Pipettor of PMC           (752)
}

def get_mover_endpoint(mover_name):
    can_id = movers[mover_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.MOVER)
    return endpoint


# CoreXY Endpoint ######################################################################################################

def get_corexy_endpoint():
    can_id = 0x0108
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.COREXY)
    return endpoint


# Measurement Endpoint #################################################################################################

def get_measurement_endpoint():
    can_id = 0x010C
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.MeaurementFunctions)
    return endpoint


# Serial Endpoint ######################################################################################################

serial_endpoints = {
    'trf'   : {'id':0x010E, 'node':'eef'},
    'bcr1'  : {'id':0x010D, 'node':'eef', 'ntrig':3},
    'bcr2'  : {'id':0x010F, 'node':'eef', 'ntrig':4},
    'bcr3'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':0},
    'bcr4'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':1},
    'bcr5'  : {'id':0x0208, 'node':'eef', 'ntrig':5, 'mux':2},
}

def get_serial_endpoint(serial_endpoint_name):
    can_id = serial_endpoints[serial_endpoint_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, EndPointType.SERIAL)
    return endpoint


# System Control Endpoint ##############################################################################################

def get_system_control_endpoint():
    can_id = 0x0120
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'SystemControlFunctions')
    return endpoint


# Fan Control Endpoint #################################################################################################

def get_fan_control_endpoint():
    can_id = 0x0121
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'FanControlFunctions')
    return endpoint


# Temperature Control Endpoint #########################################################################################

temperature_controls = {
    'eef_tc'    : {'id':0x0209},
#    'eef2_tc'   : {'id':0x02E1},
    'fmb_tc'    : {'id':0x0122},
}

def get_temperature_control_endpoint(control_name):
    can_id = temperature_controls[control_name]['id']
    endpoint = URPCFunctions.instance.endPointsDic.get(can_id)
    if endpoint is None:
        endpoint = create_endpoint(can_id, 'TemperatureControlFunctions')
    return endpoint


# Initialization #######################################################################################################

async def init(*endpoints):
    for endpoint in endpoints:
        if endpoint in nodes:
            await start_firmware(endpoint)
        if endpoint in movers:
            mover = get_mover_endpoint(endpoint)
            if endpoint == 'as1' or endpoint == 'as2':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=14, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
            elif endpoint == 'fm':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=10, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
            else:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
                )
                await mover.UseProfile(0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000, timeout=1)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0, timeout=1)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000, timeout=1)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF, timeout=1)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01, timeout=1)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1, timeout=1)
                await mover.SetConfigurationStatus(1, timeout=1)
        if endpoint == 'corexy':
            corexy = get_corexy_endpoint()
            await corexy.SetProfile(
                handle=0, speed=100000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
            )
            await corexy.UseProfile(0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RampMode,                           1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeAxisOrder,                      1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionX,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionY,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceX,              550000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceY,              550000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceX,        10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceY,        10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceX,          0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceY,          0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeCalibrationSpeed,           10000, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionX,             0x7FFFFFFF, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionY,             0x7FFFFFFF, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableX,               0x01, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableY,               0x01, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderMode,                        1, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorX,       12800, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorY,       12800, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MaxEncoderDeviation,              512, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionX,                 0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionY,                 0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionA,               0, timeout=1)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionB,               0, timeout=1)
            await corexy.SetConfigurationStatus(1, timeout=1)
        if endpoint == 'alpha_tc':
            temp_control = get_temperature_control_endpoint('eef_tc')
            await temp_control.SetParameter(0, 0, 0.33, timeout=1)
#            reference_value = int(0.33 * 1000)
#            await temp_control.SetParameter(0, 0, reference_value, timeout=1)
            await temp_control.SelectAnalogInput(channel=0, number=3, scaling=1.0, offset=0, timeout=1)
            await temp_control.SelectAnalogOutput(channel=0, number=11, scaling=1.0, offset=0, timeout=1)
            await temp_control.Configure(channel=0, dt=0.01, kp=-10.0, ti=float('inf'), td=0.0, min=0.0, max=+0.15, timeout=1)
            await temp_control.Enable(channel=0, enable=1, timeout=1)

        await asyncio.sleep(0.5)
        print(f"{endpoint} initialized")


# Mover Tests ##########################################################################################################
'''
async def fragmented_move(mover_name, position, step_size=1, timeout=1):
    mover = get_mover_endpoint(mover_name)
    current_position = (await mover.GetPosition())[0]
    print(f"current_position = {current_position}")
    if (position >= current_position):
        dir = +1
    else:
        dir = -1
    for pos in range((current_position + (step_size * dir)), (position + dir), (step_size * dir)):
        print(f"Move({pos})")
        await mover.Move(pos, timeout)
'''

# Sequencer Tests ######################################################################################################

signals = {
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}


triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

channels = {
    'pmt1'  : (0 << 24),
    'pmt2'  : (1 << 24),
    'pmt3'  : (2 << 24),
    'pmt4'  : (3 << 24),
    'ref'   : (4 << 24),
    'abs'   : (5 << 24),
    'res'   : (6 << 24),
}


async def seq_pulse_signal(name, duration_us=1):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (duration_us * 100 - 1),
        0x02000000 | signals[name],
        0x7C000000,
        0x03000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_set_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x02000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_clear_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x03000000 | signals[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_trigger(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers[name],
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.5)


async def seq_wait_for_trigger():
    meas = get_measurement_endpoint()
    sequence = [
        0x74000000 | 2,
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    done = False
    while not done:
        print(f"waiting for trigger...")
        status = (await meas.GetStatus())[0]
        if status & 0x01:
            done = True
        else:
            await asyncio.sleep(0.1)
    await asyncio.sleep(0.5)
    print(f"waiting for trigger done")


async def seq_get_analog(detector='pmt1'):
    meas = get_measurement_endpoint()
    sequence = [
        0x8C040000,                         # Clear Result Buffer
        0x01000000 | triggers[detector],    # Trigger Analog Measurement
        0x80000000 | channels[detector],    # Get Analog High Result
        0x80100001 | channels[detector],    # Get Analog Low Result
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 4, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Analog Low:  {detector} = {5 / 65536 * results[0]} V")
    print(f"Analog High: {detector} = {5 / 65536 * results[1]} V")


async def seq_get_counter(detector='pmt1', window_us=100):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    sequence = [
        0x7C000000 | (100 - 1),
        0x88B80000 | channels[detector],
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),
            0x07000000 | (65536 - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels[detector],
            0x05000000,
            0x05000000,
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),
            0x7C000000 | (100 - 1),
            0x88D00000 | channels[detector],
            0x05000000,
        ])
    sequence.extend([
        0x80900000 | channels[detector],
        0x00000000,
    ])
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 1, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Count: {detector} = {results[0]}")



async def seq_measure(window_us=1000, iterations=1):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    gate_delay = 1000000  # 10 ms
    sequence = [
        0x7C000000 | (gate_delay - 1),                          # Timer Start Gate Delay
        0x02000000 | signals['hvgate1'] | signals['hvgate2']    # Set Signals 
                   | signals['ingate1'] | signals['ingate2'] 
                   | signals['rstpmt1'] | signals['rstpmt2'],
        0x8C000000,                                             # Clear Result Buffer 0
        0x8C000001,                                             # Clear Result Buffer 1
        0x8C000002,                                             # Clear Result Buffer 2
        0x8C000003,                                             # Clear Result Buffer 3
        0x8C000004,                                             # Clear Result Buffer 4
        0x8C000005,                                             # Clear Result Buffer 5
        0x7C000000 | (100 - 1),                                 # Timer Wait And Restart 1 µs Precounter Window
        0x03000000 | signals['rstpmt1'] | signals['rstpmt2'],   # Reset Signals
        0x88B80000 | channels['pmt1'],                          # Reset Precounter and Counter
        0x88B80000 | channels['pmt2'],                          # Reset Precounter and Counter
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),                    # Loop for window_corse * 65536 µs
            0x07000000 | (65536 - 1),                           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),                             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels['pmt1'],                      # Add Precounter Value to the Counter Value
            0x88D00000 | channels['pmt2'],                      # Add Precounter Value to the Counter Value
            0x05000000,                                         # Loop End
            0x05000000,                                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),                     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),                             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels['pmt1'],                      # Add Precounter Value to the Counter Value
            0x88D00000 | channels['pmt2'],                      # Add Precounter Value to the Counter Value
            0x05000000,                                         # Loop End
        ])
    sequence.extend([
        0x01000000 | triggers['pmt1'] | triggers['pmt2'],       # Trigger Analog Measurement
        0x80900000 | channels['pmt1'],                          # Get Pulse Counter Result
        0x80000001 | channels['pmt1'],                          # Get Analog High Result
        0x80100002 | channels['pmt1'],                          # Get Analog Low Result
        0x80900003 | channels['pmt2'],                          # Get Pulse Counter Result
        0x80000004 | channels['pmt2'],                          # Get Analog High Result
        0x80100005 | channels['pmt2'],                          # Get Analog Low Result
        0x03000000 | signals['hvgate1'] | signals['hvgate2']    # Reset Signals
                   | signals['ingate1'] | signals['ingate2'],
        0x00000000,
    ])
    for i in range(0, len(sequence), 28):
        subsequence = sequence[i:(i + 28)]
        buffer = [0] * 28
        buffer[0:len(subsequence)] = subsequence
        await meas.WriteSequence(i, len(subsequence), buffer, timeout=5)

    results = [[], [], [], [], [], []]
    for i in range(iterations):
#        PyLogger.logger.info(f"seq_measure(): iteration = {i}")
        await meas.StartSequence(0, timeout=1)
        result = await meas.ReadResults(0, 6, timeout=(2 + window_us / 1000000))
        results[0].append(result[0])
        results[1].append(5 / 65536 * result[1])
        results[2].append(5 / 65536 * result[2])
        results[3].append(result[3])
        results[4].append(5 / 65536 * result[4])
        results[5].append(5 / 65536 * result[5])
    
#        PyLogger.logger.info(f"pmt1 ; count = {results[0][i]:,} ; analog_low = {results[1][i]} V ; analog_high = {results[2][i]} V")
#        PyLogger.logger.info(f"pmt2 ; count = {results[3][i]:,} ; analog_low = {results[4][i]} V ; analog_high = {results[5][i]} V")

    return results


async def seq_darkcount(iterations, delay=100000):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (delay - 1),           # TimerWaitAndRestart(delay - 1)
        0x88B80000,                         # PulseCounterControl(ch=0, add=0, RstCnt=1, RstPreCnt=1, corr=1)
        0x07000000 | (iterations - 1),      # Loop(iterations - 1)
        0x7C000000 | (delay - 1),           #     TimerWaitAndRestart(delay - 1)
        0x88D00000,                         #     PulseCounterControl(ch=0, add=1, RstCnt=0, RstPreCnt=1, corr=0)
        0x05000000,                         # LoopEnd()
        0x80900000,                         # GetPulseCounterResult(ch=0, rel=0, RstCnt=1, add=0, dword=0, addrReg=0, addr=0)
        0x00000000,                         # Stop(0)
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 1, timeout=5)
    await asyncio.sleep(0.5)
    
    print(f"Dark Count = {results[0]}")


# Serial Tests #########################################################################################################

async def serial_loopback(serial_name, *txdata):
    serial = get_serial_endpoint(serial_name)
    await serial.SetParameter(0, 1900)
    await serial.ClearReadBuffer()
    await serial.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await serial.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    await asyncio.sleep(0.5)
    print(f"rxdata: {rxdata}")


# BCR Tests ############################################################################################################
'''
async def bcr_write(bcr_name, *data):
    bcr = get_serial_endpoint(bcr_name)
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.Write(len(data), bytes(data), timeout=2)
    await asyncio.sleep(0.5)

async def bcr_read(bcr_name, length, timeout=5, sw_trigger=False):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.SetParameter(0, int(timeout * 1000))
    await bcr.ClearReadBuffer()
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31', timeout=2)
    try:
        response = await bcr.Read(length, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
    await asyncio.sleep(0.5)
    print(f"Barcode: {barcode}")

async def bcr_read_until(bcr_name, termination='\r', timeout=5, sw_trigger=False):
    if isinstance(termination, str):
        termination = ord(termination)
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.SetParameter(0, (timeout * 1000))
    await bcr.ClearReadBuffer()
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31', timeout=2)
    try:
        response = await bcr.ReadUntil(termination, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        print(ex)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
    await asyncio.sleep(0.5)
    print(f"Barcode: {barcode}")


async def bcr_overflow(bcr_name):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(6, bool(mux & 1))
        await node.SetDigitalOutput(7, bool(mux & 2))
        await node.SetDigitalOutput(8, bool(mux & 4))
    await bcr.ClearReadBuffer()
    for i in range(100):
        await node.SetDigitalOutput(ntrig, 0)
        await asyncio.sleep(0.9)
        await node.SetDigitalOutput(ntrig, 1)
        await asyncio.sleep(0.1)
    await asyncio.sleep(0.5)
    print(f"BCR Status: {await bcr.GetStatus()}")
'''

# Firmware Update Tests ################################################################################################
'''
async def fpga_update_test():
    eef = get_node_endpoint('eef')
    await eef.StartFirmwareUpdate(canID=eef.CANId, device=2, timeout=1)
    await eef.EraseFlash(canID=eef.CANId, device=2, timeout=120)
    data = range(110)
    for i in range(3):
        address = i * len(data)
        await eef.SendFirmwareData(address=address, size=len(data), data=bytes(data), timeout=5)
    await eef.EraseFlash(canID=eef.CANId, device=2, timeout=120)
    await asyncio.sleep(0.5)
'''

# Temperature Control Tests ############################################################################################
# code from Rüdiger

async def tec_log(name, channel, interval, duration):
    eef = get_node_endpoint('eef')
    tec = get_temperature_control_endpoint(name)
    file = open('temperature_control_log.txt', 'a')
    # file.write(f"TIMESTAMP               ; REFERENCE[FS] ; FEEDBACK[FS]  ; OUTPUT[FS]\n")
    file.write(f"TIMESTAMP[s]  ; REFERENCE[FS] ; FEEDBACK[FS]  ; OUTPUT[FS]\n")
    start = time.perf_counter()
    timestamp = start
    while (timestamp - start) < duration:
        results = await asyncio.gather(
            tec.GetParameter(channel, 0, timeout=1),
            tec.GetFeedbackValue(channel, timeout=1),
            tec.GetOutputValue(channel, timeout=1),
            eef.GetAnalogInput(2),
            eef.GetAnalogInput(3),
            asyncio.sleep(interval)
        )
#        reference = q31_to_float(results[0][0])
#        feedback  = q31_to_float(results[1][0])
#        output    = q31_to_float(results[2][0])
#        file.write(f"{datetime.now().isoformat(timespec='milliseconds')} ; {reference:+13.6f} ; {feedback:+13.6f} ; {output:+13.6f}\n")
#        file.write(f"{timestamp:13.3f} ; {reference:+13.6f} ; {feedback:+13.6f} ; {output:+13.6f}\n")
        reference = results[0][0]
        feedback  = results[1][0]
        output    = results[2][0]
        pd        = results[3][0]
        ntc       = results[4][0]
        file.write(f"{timestamp:13.3f}; {reference}; {feedback}; {output}; {pd}; {ntc}\n")
        timestamp = time.perf_counter()
    
    file.write(f"\n")
    file.close()


async def tec_alpha_test():
    eef = get_node_endpoint('eef')
    tec = get_temperature_control_endpoint('eef_tc')

    await init('eef', 'alpha_tc')               # Initialize EEF and Alpha TC
    await al_setcurrent( 0)                     # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await al_enable()                           # Enable laser diode current supply
    await seq_set_signal('alpha')               # Disable alpha laser
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await seq_clear_signal('alpha')             # Enable alpha laser
    await al_setcurrent( 511)                   # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await al_setcurrent( 1023)                  # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await al_setcurrent( 1535)                  # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await al_setcurrent( 2047)                  # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await al_setcurrent( 2559)                  # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await al_setcurrent( 3071)                  # Set Laser Diode Current (≈2,5V) : 3103*524288 = 1626865664
    await asyncio.sleep(10)                     # Wait 10 s to stabalize the temperature
    await seq_set_signal('alpha')               # Disable alpha laser
    await al_disable()                          # Disable laser diode current supply



async def tec_alpha_test_log():
    await asyncio.gather(
        tec_log('eef_tc', 0, 0, 100),
        tec_alpha_test()
    )


# ml Flash Functions

async def flash_setpw(pw=3515):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(2, analogscale(pw))
    await asyncio.sleep(0.1)

async def flash_highpower():
    eef = get_node_endpoint('eef')
    await eef.SetDigitalOutput(9, 1)
    await asyncio.sleep(0.1)

async def flash_highspeed():
    eef = get_node_endpoint('eef')
    await eef.SetDigitalOutput(9, 0)
    await asyncio.sleep(0.1)

# ml PMT Functions

async def pmt1_setdl(dl=1024):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(3, analogscale(dl))
    await asyncio.sleep(0.1)

async def pmt1_sethv(hv=1024):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(4, analogscale(hv))
    await asyncio.sleep(0.1)

async def pmt2_setdl(dl=1024):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(5, analogscale(dl))
    await asyncio.sleep(0.1)

async def pmt2_sethv(hv=1024):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(6, analogscale(hv))
    await asyncio.sleep(0.1)

# ml Alpha Functions

async def al_disable():
    eef = get_node_endpoint('eef')
    await eef.SetDigitalOutput(0,0)
    await asyncio.sleep(0.1)

async def al_enable():
    eef = get_node_endpoint('eef')
    await eef.SetDigitalOutput(0,1)
    await asyncio.sleep(0.1)

async def al_setcurrent(alphacurrent=3103):
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(0, analogscale(alphacurrent))
    await asyncio.sleep(0.1)

async def al_settec(tecpower=325):  # 325; 170.000.000
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(11, analogscale(tecpower))
    await asyncio.sleep(0.1)

async def al_settemp(tref=0.38):  # 0.33
    temp_control = get_temperature_control_endpoint('eef_tc')
    await temp_control.SetParameter(0, 0, tref, timeout=1)
#    reference_value = int(tref*1000)
#    await temp_control.SetParameter(0, 0, reference_value, timeout=1)
#    await temp_control.SelectAnalogInput(channel=0, number=3, scaling=1.0, offset=0, timeout=1)
#    await temp_control.SelectAnalogOutput(channel=0, number=11, scaling=1.0, offset=0, timeout=1)
#    await temp_control.Configure(channel=0, dt=0.01, kp=-10.0, ti=float('inf'), td=0.0, min=0.0, max=+0.15, timeout=1)
#    await temp_control.Enable(channel=0, enable=1, timeout=1)

async def al_settemp2(tref=0.38):  # 0.33
    temp_control = get_temperature_control_endpoint('eef_tc')
    reference_value = int(tref*1000)
    await temp_control.SetParameter(0, 0, reference_value, timeout=1)
    await temp_control.SelectAnalogInput(channel=0, number=3, scaling=1.0, offset=0, timeout=1)
    await temp_control.SelectAnalogOutput(channel=0, number=11, scaling=1.0, offset=0, timeout=1)
    await temp_control.Configure(channel=0, dt=0.01, kp=-10.0, ti=float('inf'), td=0.0, min=0.0, max=+0.15, timeout=1)
    await temp_control.Enable(channel=0, enable=1, timeout=1)

async def al_gettemp():
    eef = get_node_endpoint('eef')
    aiv = (await eef.GetAnalogInput(3))[0]
    print(f"Alpha-Temp-Value: {aiv}")
    return aiv

async def al_getpd():
    eef = get_node_endpoint('eef')
    aiv = (await eef.GetAnalogInput(2))[0]
    print(f"Alpha-PD-Value: {aiv}")
    return aiv

async def start_al(alphacurrent=511):
    await init('alpha_tc')
    await al_disable()
    await al_setcurrent(alphacurrent)
    await asyncio.sleep(0.1)

# ml TRF Serial


# TRF Laser Functions

async def seq_trf_loopback():
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers['trf'],   # Set Trigger Output
        0x74000000 | 1,                 # Wait For Trigger Input
        0x00000000,
    ]
    buffer = [0] * 28
    buffer[0:len(sequence)] = sequence
    await meas.WriteSequence(0, len(sequence), buffer, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.1)
    if (await meas.GetStatus())[0] & 0x01:
        return f"seq_trf_loopback() passed"
    else:
        return f"seq_trf_loopback() failed"

async def trf_query(command: str):
    serial = get_serial_endpoint('trf')
    command = command + '\r\n'
    await serial.SetParameter(0, 1000, timeout=1)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(command), command.encode('utf-8'), timeout=2)
    rxdata = ''
    try:
        response = await serial.ReadUntil(ord('\r'), timeout=1.1)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    return bytes(rxdata).decode('utf-8')

async def trf_send(command: str):
    serial = get_serial_endpoint('trf')
    command = command + '\r\n'
    await serial.SetParameter(0, 1000, timeout=1)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(command), command.encode('utf-8'), timeout=2)

# ml TRF Serial

async def trf_init():
    print("TRF Init")
    rxdata = await trf_query('*IDN?')
    print(f"*IDN?: {rxdata}")
    await trf_send('LD:PULS:MOD 3')     # single shot
    await trf_send('LD:PULS:FRE 2.0')   # frequency set to 2.0kHz

async def trf_enable():
    print("TRF Enable !")
    await trf_send('LD:SET:ENA ON')     # enable on

async def trf_disable():
    print("TRF Disable !")
    await trf_send('LD:SET:ENA OFF')    # enable off

async def trf_reset():
    print("TRF Reset")
    rxdata = await trf_query('*RST')    # reset
    print(f"*RST: {rxdata}")




async def mlud(mot_file):
    '''
    ml 210223:
    '''
    node = VUnits.instance.hal.nodes["EEFNode2"].node
    bootloader = VUnits.instance.hal.bootLoader
    await bootloader.FlashOneNode(node, mot_file, device=2, nodeName='EEFNode2')
    

    
